<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Imports;

use App\Monitor;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Validation\ValidationException;
use Maatwebsite\Excel\Concerns\OnEachRow;
use Maatwebsite\Excel\Concerns\SkipsEmptyRows;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Events\AfterImport;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Events\BeforeImport;
use Maatwebsite\Excel\Events\BeforeSheet;
use Maatwebsite\Excel\Row;
use Throwable;

abstract class TrackedImport implements OnEachRow, WithChunkReading, WithEvents, WithHeadingRow, SkipsEmptyRows, ShouldQueue
{
    use InteractsWithQueue;

    private Monitor $monitor;

    public function onRow(Row $row): void
    {
        if ($this->monitor->cancelled) {
            return;
        }

        $data = $row->toArray();

        try {
            $this->processRow($data);
        } catch (Throwable $t) {
            $failure = ['row' => $row->getRowIndex()] + ($t instanceof ValidationException ? ['errors' => $t->errors()] : ['error' => $t->getMessage()]);
            $this->monitor->logger()->error(json_encode($failure, JSON_PRETTY_PRINT));
            $this->monitor->failures++;
        } finally {
            $this->monitor->processed = $row->getRowIndex() - 1;
        }
    }

    abstract protected function processRow(array $data): void;

    public function registerEvents(): array
    {
        return [
            BeforeImport::class => function (BeforeImport $event) {
                $this->monitor = $this->monitor();
                $this->monitor->total = transform($event->getReader()->getTotalRows(), fn($value) => array_sum($value) - count($value));
                $this->monitor->processed = $this->monitor->failures = 0;
                $this->monitor->logger()->info('Import is queued for processing…');
            },
            BeforeSheet::class  => function (BeforeSheet $event) {
                if ($this->monitor->processed === 0) {
                    $this->monitor->logger()->info('Import process started.');
                    $this->beforeImport();
                }
            },
            AfterSheet::class   => function (AfterSheet $event) {
                if ($this->monitor->processed > 0) {
                    $this->monitor->logger()
                                  ->info("Processed: {$this->monitor->processed}/{$this->monitor->total} records, failures: {$this->monitor->failures} records.");
                }
            },
            AfterImport::class  => function () {
                if ($this->monitor->cancelled) {
                    $this->monitor->logger()->info('Import cancelled.');
                    $this->monitor->clear();
                } else {
                    $this->monitor->logger()->info('Import process finished.');
                    $this->afterImport();
                }
            }
        ];
    }

    abstract protected function monitor(): Monitor;

    protected function beforeImport()
    {
        //
    }

    protected function afterImport()
    {
        //
    }
}
