<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Jobs;

use App\Models\Campaign;
use App\Monitor;

class ProcessContactListsCampaign extends ProcessCampaign
{
    protected Campaign $campaign;

    private Monitor $monitor;

    public function __construct(Campaign $campaign)
    {
        $this->campaign = $campaign;
        $this->monitor = Monitor::for(Campaign::class, $this->campaign->id);
        $this->monitor->total = $campaign->contactLists->sum(function ($contactList) {
            return $contactList->contacts()->whereSubscribed(true)->count();
        });
        $this->monitor->processed = $this->monitor->failures = 0;
    }

    public function process(): void
    {
        $senderCount = $this->campaign->senders->count();

        foreach ($this->campaign->contactLists as $contactList) {
            $contacts = $contactList->contacts()
                                    ->with('contactFields')
                                    ->whereSubscribed(true)
                                    ->lazy();

            $proccessed = 0;
            foreach ($contacts as $contact) {
                if ($this->monitor->cancelled) {
                    $this->job->delete();
                    return;
                }

                $allowed =
                    $this->campaign->options->get('force', false) ||
                    $this->campaign->user->blacklist()
                                         ->whereMobileNumber($contact->mobile_number)
                                         ->doesntExist();

                if ($allowed) {
                    $contact->setRelation('contactList', $contactList);
                    $additional = $contact->extras + ['contact_list' => $contact->contact_list_id, 'contact' => $contact->mobile_number];
                    $senderIndex = $senderCount > 0 ? $proccessed % $senderCount : 0;
                    $this->campaign->addMessage($contact->mobile_number, $this->campaign->senders->get($senderIndex), $additional);
                    $proccessed++;
                }

                $this->monitor->processed++;
            }
        }
    }
}
