<?php
/*
 * Copyright © 2018-2025 RBSoft (Ravi Patel). All rights reserved.
 *
 * Author: Ravi Patel
 * Website: https://rbsoft.org/downloads/sms-gateway
 *
 * This software is licensed, not sold. Buyers are granted a limited, non-transferable license
 * to use this software exclusively on a single domain, subdomain, or computer. Usage on
 * multiple domains, subdomains, or computers requires the purchase of additional licenses.
 *
 * Redistribution, resale, sublicensing, or sharing of the source code, in whole or in part,
 * is strictly prohibited. Modification (except for personal use by the licensee), reverse engineering,
 * or creating derivative works based on this software is strictly prohibited.
 *
 * Unauthorized use, reproduction, or distribution of this software may result in severe civil
 * and criminal penalties and will be prosecuted to the fullest extent of the law.
 *
 * For licensing inquiries or support, please visit https://support.rbsoft.org.
 */

namespace App\Imports;

use App\Enums\FieldType;
use App\Models\ContactList;
use App\Monitor;
use App\Rules\MobileNumber;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Concerns\Importable;

class ContactsImport extends TrackedImport
{
    use Importable;

    private ContactList $contactList;

    public function __construct(ContactList $contactList)
    {
        $this->contactList = $contactList;
    }

    public function chunkSize(): int
    {
        return 1000;
    }

    protected function monitor(): Monitor
    {
        return Monitor::for(ContactsImport::class, $this->contactList->id);
    }

    /**
     * @throws \Throwable
     * @throws \Illuminate\Validation\ValidationException
     */
    protected function processRow(array $data): void
    {
        $this->contactList->fields->each(function ($field) use (&$data) {
            if ($field->type !== FieldType::Checkbox && $field->type !== FieldType::Multiselect) {
                return;
            }

            if (array_key_exists($field->tag, $data)) {
                $data[$field->tag] = transform($data[$field->tag], function ($value) {
                    return array_map('trim', str_getcsv($value));
                });
            }
        });

        Validator::make(
            data: $data,
            rules: $this->contactList->fieldsValidationRules() + [
                'mobile_number' => [
                    'bail',
                    'required',
                    new MobileNumber(),
                    Rule::unique('contacts', 'mobile_number')
                        ->where('contact_list_id', $this->contactList->id)
                ],
                'subscribed' => ['boolean']
            ]
        )->validate();

        $this->contactList->addContact($data);
    }
}
